<?php
header('Content-Type: application/json');

// --- IMPORTANT SECURITY NOTE ---
// Do NOT hardcode database credentials directly in production scripts.
// Use environment variables, a secure configuration file outside the web root,
// or a secrets management service. For demonstration, they are defined here.


if ($_SERVER['REQUEST_METHOD'] == 'POST') {

    // Retrieve and trim input data from the POST request
    $request_id = trim($_POST['request_id'] ?? '');
    $user_id = trim($_POST['user_id'] ?? '');
    $amount_str = trim($_POST['amount'] ?? '0');
    $wallet_type = trim($_POST['wallet_type'] ?? ''); // This will be "Income Balance" or "Voucher Balance"
    $new_status = trim($_POST['new_status'] ?? '');

    // Convert amount to a float after cleaning any non-numeric characters
    $cleaned_amount_str = preg_replace('/[^0-9.]/', '', $amount_str);
    $amount = !empty($cleaned_amount_str) ? floatval($cleaned_amount_str) : 0.00;

    // Basic validation for essential inputs
    if (empty($request_id) || empty($user_id) || $amount <= 0 || empty($wallet_type) || !in_array($new_status, ['Verified', 'Cancelled'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid or missing data for request update.']);
        exit; // Terminate script execution
    }

    // Validate wallet type
    if (!in_array($wallet_type, ['Income Balance', 'Voucher Balance'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid wallet type provided.']);
        exit;
    }

    // Establish database connection
   include 'dbs.php';

    // Check for connection errors
    if ($con->connect_error) {
        echo json_encode(['success' => false, 'message' => 'Database connection failed: ' . $con->connect_error]);
        exit;
    }

    // --- Start a database transaction ---
    // This ensures that either ALL operations succeed, or NONE of them do.
    $con->begin_transaction();

    try {
        // 1. Update the status of the withdraw_request
        // Only update if the current status is 'Pending' to prevent double processing
        $stmt_update_request = $con->prepare("UPDATE withdraw_request SET status = ? WHERE id = ? AND user_id = ? AND status = 'Pending'");
        if (!$stmt_update_request) {
            throw new Exception("Withdrawal request status update preparation failed: " . $con->error);
        }
        $stmt_update_request->bind_param("sis", $new_status, $request_id, $user_id);
        $stmt_update_request->execute();

        // Check if a row was actually affected (meaning the request existed and was pending)
        if ($stmt_update_request->affected_rows === 0) {
            // If affected_rows is 0, it means the request was not found, not pending, or already processed.
            throw new Exception("Withdrawal request not found, already processed, or invalid for update.");
        }
        $stmt_update_request->close();

        // 2. Deduct amount from user_list ONLY if the new_status is 'Verified'
        if ($new_status == 'Verified') {
            // Determine which balance column to update based on wallet_type
            $balance_field = '';
            if ($wallet_type == 'Income Balance') {
                $balance_field = 'balance';
            } elseif ($wallet_type == 'Voucher Balance') {
                $balance_field = 'voucher_balance';
            } else {
                // This case should ideally be caught by the earlier validation, but as a safeguard
                throw new Exception("Unknown wallet type for balance deduction.");
            }

            // Get current balance and lock the row for update to prevent race conditions
            // Using FOR UPDATE ensures no other transaction can modify this row until current transaction commits/rolls back
            $stmt_get_balance = $con->prepare("SELECT $balance_field FROM user_list WHERE user_id = ? FOR UPDATE");
            if (!$stmt_get_balance) {
                throw new Exception("Balance retrieval preparation failed: " . $con->error);
            }
            $stmt_get_balance->bind_param("s", $user_id);
            $stmt_get_balance->execute();
            $result_balance = $stmt_get_balance->get_result();

            if ($result_balance->num_rows === 0) {
                throw new Exception("User with ID '$user_id' not found in user_list.");
            }
            $current_balance = $result_balance->fetch_assoc()[$balance_field];
            $stmt_get_balance->close();

            // Check for sufficient funds before attempting deduction
            if ($current_balance < $amount) {
                throw new Exception("Insufficient balance for withdrawal. Current: " . $current_balance . ", Requested: " . $amount);
            }

            // Deduct the amount from the determined balance field
            $stmt_deduct_balance = $con->prepare("UPDATE user_list SET $balance_field = $balance_field - ? WHERE user_id = ?");
            if (!$stmt_deduct_balance) {
                throw new Exception("Balance deduction preparation failed: " . $con->error);
            }
            $stmt_deduct_balance->bind_param("ds", $amount, $user_id); // 'd' for double/float, 's' for string
            $stmt_deduct_balance->execute();

            // Check if user balance was actually updated
            if ($stmt_deduct_balance->affected_rows === 0) {
                throw new Exception("Failed to deduct amount from user balance. User ID: $user_id.");
            }
            $stmt_deduct_balance->close();
        }

        // If all operations were successful, commit the transaction
        $con->commit();
        echo json_encode(['success' => true, 'message' => 'Withdrawal request ' . $new_status . 'd and balance updated successfully.']);

    } catch (Exception $e) {
        // If any error occurs, roll back the transaction to undo all changes
        $con->rollback();
        echo json_encode(['success' => false, 'message' => 'Transaction failed: ' . $e->getMessage()]);
    } finally {
        // Always close the connection
        $con->close();
    }

} else {
    // Handle non-POST requests
    echo json_encode(['success' => false, 'message' => 'Invalid request method. Only POST allowed.']);
}
?>